/*
 * Decompiled with CFR 0.152.
 */
package fr.ens.biologie.genomique.eoulsan.util.locker;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Ordering;
import fr.ens.biologie.genomique.eoulsan.EoulsanException;
import fr.ens.biologie.genomique.eoulsan.EoulsanRuntimeException;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.ACL;
import org.apache.zookeeper.data.Stat;

@ThreadSafe
public class DistributedLock {
    private static final Logger LOG = Logger.getLogger(DistributedLock.class.getName());
    public static final int ANY_VERSION = -1;
    private final ZooKeeper zkClient;
    private final String lockPath;
    private final ImmutableList<ACL> acl;
    private final AtomicBoolean aborted = new AtomicBoolean(false);
    private CountDownLatch syncPoint;
    private boolean holdsLock = false;
    private String currentId;
    private String currentNode;
    private String watchedNode;
    private LockWatcher watcher;

    public DistributedLock(ZooKeeper zkClient, String lockPath) {
        this(zkClient, lockPath, ZooDefs.Ids.OPEN_ACL_UNSAFE);
    }

    public DistributedLock(ZooKeeper zkClient, String lockPath, Iterable<ACL> acl) {
        this.zkClient = Objects.requireNonNull(zkClient);
        this.lockPath = DistributedLock.checkNotBlank(lockPath);
        this.acl = ImmutableList.copyOf(acl);
        this.syncPoint = new CountDownLatch(1);
    }

    private synchronized void prepare() throws InterruptedException, KeeperException {
        DistributedLock.ensurePath(this.zkClient, this.acl, this.lockPath);
        LOG.log(Level.FINE, "Working with locking path:" + this.lockPath);
        this.currentNode = this.zkClient.create(this.lockPath + "/member_", null, this.acl, CreateMode.EPHEMERAL_SEQUENTIAL);
        if (this.currentNode.contains("/")) {
            this.currentId = this.currentNode.substring(this.currentNode.lastIndexOf("/") + 1);
        }
        LOG.log(Level.FINE, "Received ID from zk:" + this.currentId);
        this.watcher = new LockWatcher();
    }

    public synchronized void lock() throws IOException {
        if (this.holdsLock) {
            throw new IOException("Error, already holding a lock. Call unlock first!");
        }
        try {
            this.prepare();
            this.watcher.checkForLock();
            this.syncPoint.await();
            if (!this.holdsLock) {
                throw new IOException("Error, couldn't acquire the lock!");
            }
        }
        catch (InterruptedException e) {
            this.cancelAttempt();
            throw new IOException("InterruptedException while trying to acquire lock!", e);
        }
        catch (KeeperException e) {
            throw new IOException("KeeperException while trying to acquire lock!", e);
        }
    }

    public synchronized boolean tryLock(long timeout, TimeUnit unit) throws EoulsanException {
        if (this.holdsLock) {
            throw new EoulsanException("Error, already holding a lock. Call unlock first!");
        }
        try {
            this.prepare();
            this.watcher.checkForLock();
            boolean success = this.syncPoint.await(timeout, unit);
            if (!success) {
                return false;
            }
            if (!this.holdsLock) {
                throw new EoulsanException("Error, couldn't acquire the lock!");
            }
        }
        catch (InterruptedException e) {
            this.cancelAttempt();
            return false;
        }
        catch (KeeperException e) {
            throw new EoulsanException("KeeperException while trying to acquire lock!", e);
        }
        return true;
    }

    public synchronized void unlock() throws IOException {
        if (this.currentId == null) {
            throw new IOException("Error, neither attempting to lock nor holding a lock!");
        }
        Objects.requireNonNull(this.currentId);
        if (!this.holdsLock) {
            this.aborted.set(true);
            LOG.log(Level.INFO, "Not holding lock, aborting acquisition attempt!");
        } else {
            LOG.log(Level.INFO, "Cleaning up this locks ephemeral node.");
            this.cleanup();
        }
    }

    private synchronized void cancelAttempt() {
        LOG.log(Level.INFO, "Cancelling lock attempt!");
        this.cleanup();
        this.holdsLock = false;
        this.syncPoint.countDown();
    }

    private void cleanup() {
        LOG.info("Cleaning up!");
        Objects.requireNonNull(this.currentId);
        try {
            Stat stat = this.zkClient.exists(this.currentNode, false);
            if (stat != null) {
                this.zkClient.delete(this.currentNode, -1);
            } else {
                LOG.log(Level.WARNING, "Called cleanup but nothing to cleanup!");
            }
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.holdsLock = false;
        this.aborted.set(false);
        this.currentId = null;
        this.currentNode = null;
        this.watcher = null;
        this.syncPoint = new CountDownLatch(1);
    }

    private static void ensurePath(ZooKeeper zkClient, List<ACL> acl, String path) throws InterruptedException, KeeperException {
        Objects.requireNonNull(zkClient);
        Objects.requireNonNull(path);
        Preconditions.checkArgument((boolean)path.startsWith("/"));
        DistributedLock.ensurePathInternal(zkClient, acl, path);
    }

    private static void ensurePathInternal(ZooKeeper zkClient, List<ACL> acl, String path) throws InterruptedException, KeeperException {
        if (zkClient.exists(path, false) == null) {
            int lastPathIndex = path.lastIndexOf(47);
            if (lastPathIndex > 0) {
                DistributedLock.ensurePathInternal(zkClient, acl, path.substring(0, lastPathIndex));
            }
            try {
                zkClient.create(path, null, acl, CreateMode.PERSISTENT);
            }
            catch (KeeperException.NodeExistsException e) {
                LOG.info("Node existed when trying to ensure path " + path + ", somebody beat us to it?");
            }
        }
    }

    private static String checkNotBlank(String argument) {
        Objects.requireNonNull(argument);
        Preconditions.checkArgument((!argument.trim().isEmpty() ? 1 : 0) != 0, (Object)"Argument cannot be blank");
        return argument;
    }

    class LockWatcher
    implements Watcher {
        LockWatcher() {
        }

        public synchronized void checkForLock() {
            DistributedLock.checkNotBlank(DistributedLock.this.currentId);
            try {
                List candidates = DistributedLock.this.zkClient.getChildren(DistributedLock.this.lockPath, null);
                ImmutableList sortedMembers = Ordering.natural().immutableSortedCopy((Iterable)candidates);
                if (sortedMembers.isEmpty()) {
                    throw new EoulsanRuntimeException("Error, member list is empty!");
                }
                int memberIndex = sortedMembers.indexOf((Object)DistributedLock.this.currentId);
                if (memberIndex == 0) {
                    DistributedLock.this.holdsLock = true;
                    DistributedLock.this.syncPoint.countDown();
                } else {
                    String nextLowestNode = (String)sortedMembers.get(memberIndex - 1);
                    LOG.log(Level.INFO, String.format("Current LockWatcher with ephemeral node [%s], is waiting for [%s] to release lock.", DistributedLock.this.currentId, nextLowestNode));
                    DistributedLock.this.watchedNode = String.format("%s/%s", DistributedLock.this.lockPath, nextLowestNode);
                    Stat stat = DistributedLock.this.zkClient.exists(DistributedLock.this.watchedNode, (Watcher)this);
                    if (stat == null) {
                        this.checkForLock();
                    }
                }
            }
            catch (InterruptedException e) {
                LOG.log(Level.WARNING, String.format("Current LockWatcher with ephemeral node [%s] got interrupted. Trying to cancel lock acquisition.", DistributedLock.this.currentId), e);
                DistributedLock.this.cancelAttempt();
            }
            catch (KeeperException e) {
                LOG.log(Level.WARNING, String.format("Current LockWatcher with ephemeral node [%s] got a KeeperException. Trying to cancel lock acquisition.", DistributedLock.this.currentId), e);
                DistributedLock.this.cancelAttempt();
            }
        }

        public synchronized void process(WatchedEvent event) {
            if (!event.getPath().equals(DistributedLock.this.watchedNode)) {
                LOG.log(Level.INFO, "Ignoring call for node:" + DistributedLock.this.watchedNode);
                return;
            }
            if (event.getType() == Watcher.Event.EventType.None) {
                switch (event.getState()) {
                    case SyncConnected: {
                        LOG.info("Reconnected...");
                        break;
                    }
                    case Expired: {
                        LOG.log(Level.WARNING, String.format("Current ZK session expired![%s]", DistributedLock.this.currentId));
                        DistributedLock.this.cancelAttempt();
                    }
                }
            } else if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                this.checkForLock();
            } else {
                LOG.log(Level.WARNING, String.format("Unexpected ZK event: %s", event.getType().name()));
            }
        }
    }
}

